<?php
declare(strict_types=1);

/**
 * WHMCS API wrapper
 * Uses Identifier/Secret auth (username/password fields in legacy naming).
 * See WHMCS API docs for each action.
 */

function whmcs_config(): array {
    static $cfg = null;
    if ($cfg === null) {
        $cfg = require __DIR__ . '/../config/config.php';
    }
    return $cfg;
}

function whmcs_api(array $params): array {
    $cfg = whmcs_config();

    $payload = array_merge($params, [
        'username'     => $cfg['api_identifier'],
        'password'     => $cfg['api_secret'],
        'responsetype' => 'json',
    ]);

    $ch = curl_init($cfg['whmcs_api_url']);
    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => http_build_query($payload),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 25,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) {
        return ['result' => 'error', 'message' => 'cURL error: ' . $err, 'httpcode' => $code];
    }

    $decoded = json_decode($resp, true);
    if (!is_array($decoded)) {
        return ['result' => 'error', 'message' => 'Invalid JSON from WHMCS API', 'raw' => $resp, 'httpcode' => $code];
    }

    return $decoded;
}

/** Auth */
function whmcs_validate_login(string $email, string $password): array {
    return whmcs_api([
        'action'    => 'ValidateLogin',
        'email'     => $email,
        'password2' => $password,
    ]);
}

function whmcs_get_clients_details(int $clientId): array {
    return whmcs_api([
        'action'   => 'GetClientsDetails',
        'clientid' => (string)$clientId,
        'stats'    => 'true',
    ]);
}

function whmcs_create_sso(int $clientId, string $destination = ''): array {
    $params = [
        'action'    => 'CreateSsoToken',
        'client_id' => (string)$clientId,
    ];
    if ($destination !== '') $params['destination'] = $destination;
    return whmcs_api($params);
}

/** Invoices */
function whmcs_get_invoices(int $clientId, int $limit = 50, int $offset = 0, string $status=''): array {
    $p = [
        'action'   => 'GetInvoices',
        'userid'   => (string)$clientId,
        'limitnum' => (string)$limit,
        'limitstart' => (string)$offset,
    ];
    if ($status !== '') $p['status'] = $status;
    return whmcs_api($p);
}

function whmcs_get_invoice(int $invoiceId): array {
    return whmcs_api([
        'action'    => 'GetInvoice',
        'invoiceid' => (string)$invoiceId,
    ]);
}

/** Services */
function whmcs_get_clients_products(int $clientId, int $limit=50, int $offset=0, string $status=''): array {
    $p = [
        'action'     => 'GetClientsProducts',
        'clientid'   => (string)$clientId,
        'limitnum'   => (string)$limit,
        'limitstart' => (string)$offset,
    ];
    if ($status !== '') $p['status'] = $status;
    return whmcs_api($p);
}

function whmcs_module_command(int $serviceId, string $command): array {
    return whmcs_api([
        'action'   => 'Module' . $command,
        'serviceid'=> (string)$serviceId,
    ]);
}

/** Tickets */
function whmcs_get_tickets(int $clientId, int $limit=50, int $offset=0, string $status=''): array {
    $p = [
        'action'     => 'GetTickets',
        'clientid'   => (string)$clientId,
        'limitnum'   => (string)$limit,
        'limitstart' => (string)$offset,
    ];
    if ($status !== '') $p['status'] = $status;
    return whmcs_api($p);
}

function whmcs_get_ticket(int $ticketId): array {
    return whmcs_api([
        'action'   => 'GetTicket',
        'ticketid' => (string)$ticketId,
    ]);
}

function whmcs_add_ticket_reply(int $ticketId, string $message, string $markdown=''): array {
    return whmcs_api([
        'action'   => 'AddTicketReply',
        'ticketid' => (string)$ticketId,
        'message'  => $message,
        'markdown' => $markdown, // optional; some versions ignore
    ]);
}

function whmcs_open_ticket(int $clientId, string $deptid, string $subject, string $message, string $priority='Medium'): array {
    return whmcs_api([
        'action'   => 'OpenTicket',
        'clientid' => (string)$clientId,
        'deptid'   => $deptid,
        'subject'  => $subject,
        'message'  => $message,
        'priority' => $priority,
    ]);
}


/** Profile updates */
function whmcs_update_client(int $clientId, array $fields): array {
    $p = array_merge([
        'action'   => 'UpdateClient',
        'clientid' => (string)$clientId,
    ], $fields);
    return whmcs_api($p);
}

function whmcs_update_client_password(int $clientId, string $newPassword): array {
    return whmcs_api([
        'action'   => 'UpdateClientPassword',
        'clientid' => (string)$clientId,
        'password2'=> $newPassword,
    ]);
}
